package malasada

import (
	"bytes"
	"crypto/rand"
	"debug/elf"
	_ "embed"
	"encoding/binary"
	"errors"
	"fmt"
	"os"
	"strings"

	keystone "github.com/moloch--/go-keystone"
	"github.com/sliverarmory/malasada/internal/aplib"
)

// Arch is a linux architecture supported by malasada.
type Arch int

const (
	ArchUnknown Arch = iota
	ArchLinuxAMD64
	ArchLinuxARM64
)

func (a Arch) String() string {
	switch a {
	case ArchLinuxAMD64:
		return "linux/amd64"
	case ArchLinuxARM64:
		return "linux/arm64"
	default:
		return "unknown"
	}
}

func archFromELFMachine(m elf.Machine) (Arch, error) {
	switch m {
	case elf.EM_X86_64:
		return ArchLinuxAMD64, nil
	case elf.EM_AARCH64:
		return ArchLinuxARM64, nil
	default:
		return ArchUnknown, fmt.Errorf("unsupported ELF machine %v", m)
	}
}

var (
	errBadELF = errors.New("bad ELF")
)

// DefaultCallExport is the default exported symbol name called by the generated
// entry stub when the caller does not specify one.
const DefaultCallExport = "StartW"

// ConvertSharedObject reads a Linux ELF shared object from soPath, patches it to
// call exportName as the process entrypoint, and wraps it with the stage0 loader
// to produce an executable PIC .bin blob.
func ConvertSharedObject(soPath string, exportName string, compress bool) ([]byte, error) {
	if exportName == "" {
		exportName = DefaultCallExport
	}
	so, err := os.ReadFile(soPath)
	if err != nil {
		return nil, err
	}
	// Common footgun: Go's buildmode=c-archive produces an ar archive (.a), which
	// is sometimes misnamed as ".so". Provide a clear error early.
	if bytes.HasPrefix(so, []byte("!<arch>\n")) {
		return nil, fmt.Errorf("input %q is an ar archive (likely built with -buildmode=c-archive); malasada requires an ELF shared object (.so). Rebuild with: go build -buildmode=c-shared -o payload.so <pkg>", soPath)
	}

	arch, patchedSO, err := patchSOToCallExport(so, exportName)
	if err != nil {
		return nil, err
	}

	payload := patchedSO
	if compress {
		payload, err = aplib.PackSafe(patchedSO)
		if err != nil {
			return nil, err
		}
	}

	stage0, err := buildStage0(arch)
	if err != nil {
		return nil, err
	}
	if err := patchStage0PayloadLen(stage0, uint64(len(payload))); err != nil {
		return nil, err
	}

	out := make([]byte, 0, len(stage0)+len(payload))
	out = append(out, stage0...)
	out = append(out, payload...)
	return out, nil
}

// -----------------------------
// stage0 build + patching
// -----------------------------

const msdaMagicLen = 8

var msdaMagic = newMSDAMagic()

func newMSDAMagic() []byte {
	const letters = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
	if len(letters) != 52 {
		// If this ever changes, the rejection-sampling below must be updated.
		panic("msda magic: unexpected alphabet length")
	}

	// 52*4 is the largest multiple of 52 < 256, so we can map bytes uniformly.
	const cutoff = 52 * 4 // 208

	out := make([]byte, msdaMagicLen)
	out[0] = 'M'
	for i := 1; i < len(out); {
		var buf [32]byte
		if _, err := rand.Read(buf[:]); err != nil {
			// Extremely unlikely; fall back to the historical magic.
			copy(out, []byte("MALASADA"))
			return out
		}
		for _, b := range buf {
			if b >= cutoff {
				continue
			}
			out[i] = letters[int(b)%len(letters)]
			i++
			if i == len(out) {
				break
			}
		}
	}
	return out
}

//go:generate go run ./internal/stage0/genstage0 -out internal/stage0

// Prebuilt stage0 blobs (generated by `go generate`). These are embedded so the
// CLI can convert .so -> .bin without needing zig at runtime.
//
// IMPORTANT: Always copy before mutating; embed-backed bytes may live in
// read-only memory.
//
//go:embed internal/stage0/stage0_linux_amd64.bin
var stage0LinuxAMD64Prebuilt []byte

//go:embed internal/stage0/stage0_linux_arm64.bin
var stage0LinuxARM64Prebuilt []byte

func patchStage0PayloadLen(stage0 []byte, payloadLen uint64) error {
	// The stage0 linker script forces the msda header to the end of the extracted
	// .text. We enforce that invariant so we can append the payload bytes directly
	// after the header.
	const headerSize = 8 + 4 + 4 + 8
	if len(stage0) < headerSize {
		return fmt.Errorf("stage0 too small")
	}

	off := len(stage0) - headerSize
	hdr := stage0[off:]
	if binary.LittleEndian.Uint32(hdr[8:12]) != 1 {
		return fmt.Errorf("stage0 has unexpected msda version")
	}
	arch := binary.LittleEndian.Uint32(hdr[12:16])
	if arch != 1 && arch != 2 {
		return fmt.Errorf("stage0 has unexpected msda arch")
	}

	if len(msdaMagic) != 8 {
		return fmt.Errorf("internal error: msdaMagic length is %d, want 8", len(msdaMagic))
	}
	copy(hdr[0:8], msdaMagic)

	// payload_len is at offset 16 from magic start.
	binary.LittleEndian.PutUint64(hdr[16:24], payloadLen)
	return nil
}

func buildStage0(arch Arch) ([]byte, error) {
	var prebuilt []byte
	switch arch {
	case ArchLinuxAMD64:
		prebuilt = stage0LinuxAMD64Prebuilt
	case ArchLinuxARM64:
		prebuilt = stage0LinuxARM64Prebuilt
	default:
		return nil, fmt.Errorf("unsupported arch %v", arch)
	}
	if len(prebuilt) == 0 {
		return nil, fmt.Errorf("missing embedded stage0 for %v (run `go generate ./...`)", arch)
	}
	stage0 := make([]byte, len(prebuilt))
	copy(stage0, prebuilt)
	return stage0, nil
}

// -----------------------------
// .so patching (call export)
// -----------------------------

const (
	elf64HeaderSize = 64
	elf64PhdrSize   = 56

	ptLoad    = 1
	ptDynamic = 2
	ptInterp  = 3
	ptNote    = 4
	ptPhdr    = 6

	ptGnuEhFrame = 0x6474e550
	ptGnuStack   = 0x6474e551
)

const (
	pfX = 0x1
	pfW = 0x2
	pfR = 0x4
)

const (
	dtNull   = 0
	dtFlags1 = 0x6ffffffb

	dtInit           = 12
	dtInitArray      = 25
	dtInitArraySz    = 27
	dtPreinitArray   = 32
	dtPreinitArraySz = 33
)

const (
	df1Pie = 0x08000000
)

type elf64Ehdr struct {
	ident      [16]byte
	typ        uint16
	machine    uint16
	version    uint32
	entry      uint64
	phoff      uint64
	shoff      uint64
	flags      uint32
	ehsize     uint16
	phentsize  uint16
	phnum      uint16
	shentsize  uint16
	shnum      uint16
	shstrndx   uint16
	_entryOff  int
	_phoffOff  int
	_phnumOff  int
	_phentsOff int
}

type elf64Phdr struct {
	typ    uint32
	flags  uint32
	off    uint64
	vaddr  uint64
	paddr  uint64
	filesz uint64
	memsz  uint64
	align  uint64

	fileOff int // where this phdr starts in the file
}

func parseELF64Header(b []byte) (*elf64Ehdr, error) {
	if len(b) < elf64HeaderSize {
		return nil, errBadELF
	}
	var h elf64Ehdr
	copy(h.ident[:], b[:16])
	if h.ident[0] != 0x7f || h.ident[1] != 'E' || h.ident[2] != 'L' || h.ident[3] != 'F' {
		return nil, errBadELF
	}
	if h.ident[4] != 2 { // ELFCLASS64
		return nil, fmt.Errorf("%w: not ELF64", errBadELF)
	}
	if h.ident[5] != 1 { // ELFDATA2LSB
		return nil, fmt.Errorf("%w: not little-endian", errBadELF)
	}
	h.typ = binary.LittleEndian.Uint16(b[16:18])
	h.machine = binary.LittleEndian.Uint16(b[18:20])
	h.version = binary.LittleEndian.Uint32(b[20:24])
	h.entry = binary.LittleEndian.Uint64(b[24:32])
	h.phoff = binary.LittleEndian.Uint64(b[32:40])
	h.shoff = binary.LittleEndian.Uint64(b[40:48])
	h.flags = binary.LittleEndian.Uint32(b[48:52])
	h.ehsize = binary.LittleEndian.Uint16(b[52:54])
	h.phentsize = binary.LittleEndian.Uint16(b[54:56])
	h.phnum = binary.LittleEndian.Uint16(b[56:58])
	h.shentsize = binary.LittleEndian.Uint16(b[58:60])
	h.shnum = binary.LittleEndian.Uint16(b[60:62])
	h.shstrndx = binary.LittleEndian.Uint16(b[62:64])

	h._entryOff = 24
	h._phoffOff = 32
	h._phentsOff = 54
	h._phnumOff = 56

	if h.ehsize != elf64HeaderSize {
		return nil, fmt.Errorf("%w: unexpected ehsize %d", errBadELF, h.ehsize)
	}
	if h.phentsize != elf64PhdrSize {
		return nil, fmt.Errorf("%w: unexpected phentsize %d", errBadELF, h.phentsize)
	}
	if h.phoff == 0 || h.phnum == 0 {
		return nil, fmt.Errorf("%w: missing program headers", errBadELF)
	}
	phEnd := h.phoff + uint64(h.phnum)*uint64(h.phentsize)
	if phEnd > uint64(len(b)) {
		return nil, fmt.Errorf("%w: phdrs outside file", errBadELF)
	}
	return &h, nil
}

func parseELF64Phdrs(b []byte, h *elf64Ehdr) ([]elf64Phdr, error) {
	out := make([]elf64Phdr, 0, h.phnum)
	off := int(h.phoff)
	for i := 0; i < int(h.phnum); i++ {
		p := b[off+i*elf64PhdrSize : off+(i+1)*elf64PhdrSize]
		ph := elf64Phdr{
			typ:     binary.LittleEndian.Uint32(p[0:4]),
			flags:   binary.LittleEndian.Uint32(p[4:8]),
			off:     binary.LittleEndian.Uint64(p[8:16]),
			vaddr:   binary.LittleEndian.Uint64(p[16:24]),
			paddr:   binary.LittleEndian.Uint64(p[24:32]),
			filesz:  binary.LittleEndian.Uint64(p[32:40]),
			memsz:   binary.LittleEndian.Uint64(p[40:48]),
			align:   binary.LittleEndian.Uint64(p[48:56]),
			fileOff: off + i*elf64PhdrSize,
		}
		out = append(out, ph)
	}
	return out, nil
}

func writeELF64Entry(b []byte, entry uint64) {
	binary.LittleEndian.PutUint64(b[24:32], entry)
}

func writeELF64Phdr(b []byte, ph elf64Phdr) {
	p := b[ph.fileOff : ph.fileOff+elf64PhdrSize]
	binary.LittleEndian.PutUint32(p[0:4], ph.typ)
	binary.LittleEndian.PutUint32(p[4:8], ph.flags)
	binary.LittleEndian.PutUint64(p[8:16], ph.off)
	binary.LittleEndian.PutUint64(p[16:24], ph.vaddr)
	binary.LittleEndian.PutUint64(p[24:32], ph.paddr)
	binary.LittleEndian.PutUint64(p[32:40], ph.filesz)
	binary.LittleEndian.PutUint64(p[40:48], ph.memsz)
	binary.LittleEndian.PutUint64(p[48:56], ph.align)
}

// swapELF64PhdrContents swaps the payload-relevant fields of a and b, but keeps
// their fileOff values unchanged (fileOff represents the slot/location in the
// program header table).
func swapELF64PhdrContents(a, b *elf64Phdr) {
	a.typ, b.typ = b.typ, a.typ
	a.flags, b.flags = b.flags, a.flags
	a.off, b.off = b.off, a.off
	a.vaddr, b.vaddr = b.vaddr, a.vaddr
	a.paddr, b.paddr = b.paddr, a.paddr
	a.filesz, b.filesz = b.filesz, a.filesz
	a.memsz, b.memsz = b.memsz, a.memsz
	a.align, b.align = b.align, a.align
}

func alignUp(x, align uint64) uint64 {
	if align == 0 {
		return x
	}
	return (x + (align - 1)) &^ (align - 1)
}

func patchSOToCallExport(so []byte, exportName string) (Arch, []byte, error) {
	// Use debug/elf for symbol resolution (parsing only).
	f, err := elf.NewFile(bytes.NewReader(so))
	if err != nil {
		return ArchUnknown, nil, err
	}
	if f.Class != elf.ELFCLASS64 || f.Data != elf.ELFDATA2LSB {
		return ArchUnknown, nil, fmt.Errorf("unsupported ELF (need 64-bit little-endian)")
	}
	if f.Type != elf.ET_DYN {
		return ArchUnknown, nil, fmt.Errorf("expected ET_DYN (.so), got %v", f.Type)
	}

	arch, err := archFromELFMachine(f.Machine)
	if err != nil {
		return ArchUnknown, nil, err
	}

	dynSyms, err := f.DynamicSymbols()
	if err != nil {
		return ArchUnknown, nil, err
	}
	var exportVaddr uint64
	found := false
	for _, s := range dynSyms {
		if s.Name == exportName {
			exportVaddr = s.Value
			found = true
			break
		}
	}
	if !found {
		return ArchUnknown, nil, fmt.Errorf("export %q not found in .dynsym", exportName)
	}

	h, err := parseELF64Header(so)
	if err != nil {
		return ArchUnknown, nil, err
	}
	if elf.Type(h.typ) != elf.ET_DYN {
		return ArchUnknown, nil, fmt.Errorf("%w: not ET_DYN", errBadELF)
	}
	if elf.Machine(h.machine) != f.Machine {
		return ArchUnknown, nil, fmt.Errorf("%w: machine mismatch", errBadELF)
	}
	phdrs, err := parseELF64Phdrs(so, h)
	if err != nil {
		return ArchUnknown, nil, err
	}

	// Find a program header we can repurpose into a new executable PT_LOAD for the entry stub.
	stubPhIdx := -1
	for i := range phdrs {
		if phdrs[i].typ == ptNote {
			stubPhIdx = i
			break
		}
	}
	if stubPhIdx < 0 {
		for i := range phdrs {
			if phdrs[i].typ == ptGnuEhFrame {
				stubPhIdx = i
				break
			}
		}
	}
	if stubPhIdx < 0 {
		return ArchUnknown, nil, fmt.Errorf("no suitable program header to repurpose (need PT_NOTE or PT_GNU_EH_FRAME)")
	}

	// Find a program header we can repurpose into PT_INTERP so glibc's dynamic
	// loader treats the payload like a normal PIE executable.
	interpPhIdx := -1
	for i := range phdrs {
		if i == stubPhIdx {
			continue
		}
		if phdrs[i].typ == ptGnuStack {
			interpPhIdx = i
			break
		}
	}
	if interpPhIdx < 0 {
		for i := range phdrs {
			if i == stubPhIdx {
				continue
			}
			// Fall back to repurposing another low-value header if needed.
			if phdrs[i].typ == ptNote || phdrs[i].typ == ptGnuEhFrame {
				interpPhIdx = i
				break
			}
		}
	}
	if interpPhIdx < 0 {
		return ArchUnknown, nil, fmt.Errorf("no suitable program header to repurpose for PT_INTERP (need PT_GNU_STACK, PT_NOTE, or PT_GNU_EH_FRAME)")
	}

	// Ensure the payload has a PT_PHDR entry. Some glibc rtld code paths rely on
	// this when computing the executable's load bias.
	phdrIdx := -1
	for i := range phdrs {
		if phdrs[i].typ == ptPhdr {
			phdrIdx = i
			break
		}
	}
	phdrPhIdx := -1
	if phdrIdx < 0 {
		for i := range phdrs {
			if i == stubPhIdx || i == interpPhIdx {
				continue
			}
			if phdrs[i].typ == ptGnuEhFrame {
				phdrPhIdx = i
				break
			}
		}
		if phdrPhIdx < 0 {
			return ArchUnknown, nil, fmt.Errorf("no suitable program header to repurpose for PT_PHDR (need PT_PHDR or PT_GNU_EH_FRAME)")
		}
	}

	var maxVaddrEnd uint64
	var maxAlign uint64 = 0x1000
	var dynPh *elf64Phdr
	for i := range phdrs {
		ph := &phdrs[i]
		if ph.typ == ptLoad {
			end := ph.vaddr + ph.memsz
			if end > maxVaddrEnd {
				maxVaddrEnd = end
			}
			if ph.align > maxAlign {
				maxAlign = ph.align
			}
		}
		if ph.typ == ptDynamic {
			dynPh = ph
		}
	}
	if maxVaddrEnd == 0 {
		return ArchUnknown, nil, fmt.Errorf("%w: no PT_LOAD segments", errBadELF)
	}
	if dynPh == nil {
		return ArchUnknown, nil, fmt.Errorf("%w: no PT_DYNAMIC segment", errBadELF)
	}

	stubFileOff := alignUp(uint64(len(so)), maxAlign)
	stubVaddr := alignUp(maxVaddrEnd, maxAlign)

	initVaddrs, err := extractInitVaddrs(so, phdrs, *dynPh)
	if err != nil {
		return ArchUnknown, nil, err
	}

	// Append padding then the entry stub itself.
	so2 := make([]byte, 0, int(stubFileOff)+1024)
	so2 = append(so2, so...)
	if pad := int(stubFileOff) - len(so2); pad > 0 {
		so2 = append(so2, bytes.Repeat([]byte{0}, pad)...)
	}

	stub, err := makeEntryStub(arch, stubVaddr, exportVaddr, initVaddrs)
	if err != nil {
		return ArchUnknown, nil, err
	}
	so2 = append(so2, stub...)

	// Add a PT_INTERP string (within the stub PT_LOAD) so glibc ld-linux treats
	// the payload like a normal executable (needed for some rtld invariants).
	interpPath := ""
	switch arch {
	case ArchLinuxAMD64:
		interpPath = "/lib64/ld-linux-x86-64.so.2"
	case ArchLinuxARM64:
		interpPath = "/lib/ld-linux-aarch64.so.1"
	default:
		return ArchUnknown, nil, fmt.Errorf("unsupported arch %v", arch)
	}
	interpStr := append([]byte(interpPath), 0)
	interpFileOff := uint64(len(so2))
	interpVaddr := stubVaddr + uint64(len(stub))
	so2 = append(so2, interpStr...)

	// Convert the selected phdr into our stub PT_LOAD.
	stubPh := phdrs[stubPhIdx]
	stubPh.typ = ptLoad
	stubPh.flags = pfR | pfX
	stubPh.off = stubFileOff
	stubPh.vaddr = stubVaddr
	stubPh.paddr = stubVaddr
	stubPh.filesz = uint64(len(stub) + len(interpStr))
	stubPh.memsz = uint64(len(stub) + len(interpStr))
	stubPh.align = maxAlign
	phdrs[stubPhIdx] = stubPh

	// Repurpose another phdr into PT_INTERP.
	interpPh := phdrs[interpPhIdx]
	interpPh.typ = ptInterp
	interpPh.flags = pfR
	interpPh.off = interpFileOff
	interpPh.vaddr = interpVaddr
	interpPh.paddr = interpVaddr
	interpPh.filesz = uint64(len(interpStr))
	interpPh.memsz = uint64(len(interpStr))
	interpPh.align = 1
	phdrs[interpPhIdx] = interpPh

	// Add PT_PHDR if we had to repurpose one, and ensure it is the first phdr.
	if phdrPhIdx >= 0 {
		phdrPh := phdrs[phdrPhIdx]
		phdrPh.typ = ptPhdr
		phdrPh.flags = pfR
		phdrPh.off = h.phoff
		phdrPh.vaddr = h.phoff
		phdrPh.paddr = h.phoff
		phdrPh.filesz = uint64(h.phnum) * uint64(h.phentsize)
		phdrPh.memsz = phdrPh.filesz
		phdrPh.align = 8
		phdrs[phdrPhIdx] = phdrPh
		phdrIdx = phdrPhIdx
	}
	if phdrIdx >= 0 && phdrIdx != 0 {
		// glibc's rtld startup code expects to discover the load bias from PT_PHDR
		// before processing PT_DYNAMIC for the main program. Put PT_PHDR in slot 0
		// to match normal PIE executables.
		swapELF64PhdrContents(&phdrs[0], &phdrs[phdrIdx])
		phdrIdx = 0
	}

	// Patch e_entry to point at the stub.
	writeELF64Entry(so2, stubVaddr)

	// Write back all program headers (some are repurposed/reordered above).
	for i := range phdrs {
		writeELF64Phdr(so2, phdrs[i])
	}

	// Patch DT_FLAGS_1 |= DF_1_PIE if present.
	if err := patchDTFlags1Pie(so2, *dynPh); err != nil {
		return ArchUnknown, nil, err
	}

	return arch, so2, nil
}

func patchDTFlags1Pie(so []byte, dynPh elf64Phdr) error {
	// dynPh.off points to the dynamic entries in the file.
	if dynPh.off+dynPh.filesz > uint64(len(so)) {
		return fmt.Errorf("%w: PT_DYNAMIC outside file", errBadELF)
	}
	dyn := so[dynPh.off : dynPh.off+dynPh.filesz]
	if len(dyn)%16 != 0 {
		// Not fatal; still try to parse what we can.
	}
	for off := 0; off+16 <= len(dyn); off += 16 {
		tag := int64(binary.LittleEndian.Uint64(dyn[off : off+8]))
		val := binary.LittleEndian.Uint64(dyn[off+8 : off+16])
		if tag == dtNull {
			break
		}
		if uint64(tag) == dtFlags1 {
			val |= df1Pie
			binary.LittleEndian.PutUint64(dyn[off+8:off+16], val)
			return nil
		}
	}
	// If there's no DT_FLAGS_1, we still proceed; glibc will treat it as a normal
	// shared object, but for our exec-like jump into ld-linux we primarily need
	// a non-zero e_entry.
	return nil
}

func vaddrToFileOff(phdrs []elf64Phdr, vaddr uint64, size uint64) (uint64, error) {
	for i := range phdrs {
		ph := phdrs[i]
		if ph.typ != ptLoad {
			continue
		}
		// For file-backed data (like init arrays), we need it to be within the
		// segment's file image, not just its in-memory range.
		if vaddr < ph.vaddr || vaddr+size > ph.vaddr+ph.filesz {
			continue
		}
		off := ph.off + (vaddr - ph.vaddr)
		return off, nil
	}
	return 0, fmt.Errorf("%w: vaddr 0x%x not in any PT_LOAD file range", errBadELF, vaddr)
}

func extractInitVaddrs(so []byte, phdrs []elf64Phdr, dynPh elf64Phdr) ([]uint64, error) {
	// dynPh.off points to the dynamic entries in the file.
	if dynPh.off+dynPh.filesz > uint64(len(so)) {
		return nil, fmt.Errorf("%w: PT_DYNAMIC outside file", errBadELF)
	}
	dyn := so[dynPh.off : dynPh.off+dynPh.filesz]

	var (
		initVaddr         uint64
		preinitArrayVaddr uint64
		preinitArraySz    uint64
		initArrayVaddr    uint64
		initArraySz       uint64
	)

	for off := 0; off+16 <= len(dyn); off += 16 {
		tag := int64(binary.LittleEndian.Uint64(dyn[off : off+8]))
		val := binary.LittleEndian.Uint64(dyn[off+8 : off+16])
		if tag == dtNull {
			break
		}
		switch uint64(tag) {
		case dtInit:
			initVaddr = val
		case dtPreinitArray:
			preinitArrayVaddr = val
		case dtPreinitArraySz:
			preinitArraySz = val
		case dtInitArray:
			initArrayVaddr = val
		case dtInitArraySz:
			initArraySz = val
		}
	}

	readPtrArray := func(arrayVaddr, arraySz uint64) ([]uint64, error) {
		if arrayVaddr == 0 || arraySz == 0 {
			return nil, nil
		}
		if arraySz%8 != 0 {
			return nil, fmt.Errorf("%w: init array size 0x%x not 8-byte aligned", errBadELF, arraySz)
		}
		fileOff, err := vaddrToFileOff(phdrs, arrayVaddr, arraySz)
		if err != nil {
			return nil, err
		}
		if fileOff+arraySz > uint64(len(so)) {
			return nil, fmt.Errorf("%w: init array outside file", errBadELF)
		}
		b := so[fileOff : fileOff+arraySz]
		out := make([]uint64, 0, arraySz/8)
		for i := uint64(0); i < arraySz; i += 8 {
			ptr := binary.LittleEndian.Uint64(b[i : i+8])
			if ptr != 0 {
				out = append(out, ptr)
			}
		}
		return out, nil
	}

	var initCalls []uint64
	// Match glibc's overall ordering:
	// - DT_PREINIT_ARRAY (executables)
	// - DT_INIT
	// - DT_INIT_ARRAY
	pre, err := readPtrArray(preinitArrayVaddr, preinitArraySz)
	if err != nil {
		return nil, err
	}
	initCalls = append(initCalls, pre...)

	if initVaddr != 0 {
		initCalls = append(initCalls, initVaddr)
	}

	initArr, err := readPtrArray(initArrayVaddr, initArraySz)
	if err != nil {
		return nil, err
	}
	initCalls = append(initCalls, initArr...)

	return initCalls, nil
}

func makeEntryStub(arch Arch, stubVaddr uint64, exportVaddr uint64, initVaddrs []uint64) ([]byte, error) {
	switch arch {
	case ArchLinuxAMD64:
		return makeStubAMD64(stubVaddr, exportVaddr, initVaddrs)
	case ArchLinuxARM64:
		return makeStubARM64(stubVaddr, exportVaddr, initVaddrs)
	default:
		return nil, fmt.Errorf("unsupported arch %v", arch)
	}
}

func makeStubAMD64(stubVaddr uint64, exportVaddr uint64, initVaddrs []uint64) ([]byte, error) {
	if stubVaddr+5 > 0xffffffff {
		// Keep the encoding simple (sub imm32).
		return nil, fmt.Errorf("stub vaddr too large for amd64 stub encoding")
	}

	// Assemble the stub from textual assembly to keep it readable and reduce
	// manual byte encoding errors. We use explicit forms (like movabs) to keep
	// output byte-for-byte compatible with the historical encoding.
	var sb strings.Builder
	sb.WriteString(".code64\n")

	// Base computation:
	// call next; pop rbx; sub rbx, (stub_vaddr+5) => rbx becomes the module base.
	// rbx = (retaddr == stub+5) - (stub_vaddr+5)
	sb.WriteString("call 5\n")
	sb.WriteString("pop rbx\n")
	fmt.Fprintf(&sb, "sub rbx, 0x%X\n", stubVaddr+5)

	// Preserve entry stack pointer and compute (argc, argv, envp) once.
	// r15 = entry_rsp
	// r12 = argc
	// r13 = argv
	// r14 = envp
	sb.WriteString("mov r15, rsp\n")
	sb.WriteString("mov r12, [r15]\n")
	sb.WriteString("lea r13, [r15+8]\n")
	sb.WriteString("mov rcx, r12\n")
	sb.WriteString("add rcx, 2\n")
	sb.WriteString("shl rcx, 3\n")
	sb.WriteString("lea r14, [r15+rcx]\n")

	// Call init functions (DT_PREINIT_ARRAY/DT_INIT/DT_INIT_ARRAY).
	for _, vaddr := range initVaddrs {
		if vaddr == 0 {
			continue
		}
		// rdi = argc, rsi = argv, rdx = envp
		sb.WriteString("mov rdi, r12\n")
		sb.WriteString("mov rsi, r13\n")
		sb.WriteString("mov rdx, r14\n")
		fmt.Fprintf(&sb, "movabs rax, 0x%X\n", vaddr)
		sb.WriteString("add rax, rbx\n")
		sb.WriteString("call rax\n")
	}

	// Call the requested export (no args).
	fmt.Fprintf(&sb, "movabs rax, 0x%X\n", exportVaddr)
	sb.WriteString("add rax, rbx\n")
	sb.WriteString("call rax\n")

	// exit_group(0)
	sb.WriteString("mov eax, 231\n")
	sb.WriteString("xor edi, edi\n")
	sb.WriteString("syscall\n")

	engine, err := keystone.NewEngine(keystone.ARCH_X86, keystone.MODE_64)
	if err != nil {
		return nil, err
	}
	defer func() { _ = engine.Close() }()
	if err := engine.Option(keystone.OPT_SYNTAX, keystone.OPT_SYNTAX_INTEL); err != nil {
		return nil, err
	}
	return engine.Assemble(sb.String(), 0)
}

func makeStubARM64(stubVaddr uint64, exportVaddr uint64, initVaddrs []uint64) ([]byte, error) {
	// See internal notes in stage0: x19 = stub_start; base = x19 - stub_vaddr.
	// Then call init functions (DT_INIT/DT_INIT_ARRAY) and the requested export.

	var sb strings.Builder
	sb.WriteString("stub_start:\n")
	sb.WriteString("adr x19, stub_start\n")

	// x20 = stub_vaddr
	fmt.Fprintf(&sb, "movz x20, #0x%X\n", uint16(stubVaddr>>0))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #16\n", uint16(stubVaddr>>16))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #32\n", uint16(stubVaddr>>32))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #48\n", uint16(stubVaddr>>48))
	sb.WriteString("sub x19, x19, x20\n")

	// Compute (argc, argv, envp) from the entry stack.
	// x21 = argc
	// x22 = argv (points to argv[0])
	// x23 = envp
	sb.WriteString("ldr x21, [sp]\n")
	sb.WriteString("add x22, sp, #8\n")
	sb.WriteString("add x23, x22, x21, lsl #3\n")
	sb.WriteString("add x23, x23, #8\n")

	// Call init functions (DT_PREINIT_ARRAY/DT_INIT/DT_INIT_ARRAY).
	for _, vaddr := range initVaddrs {
		if vaddr == 0 {
			continue
		}
		sb.WriteString("mov x0, x21\n")
		sb.WriteString("mov x1, x22\n")
		sb.WriteString("mov x2, x23\n")
		fmt.Fprintf(&sb, "movz x20, #0x%X\n", uint16(vaddr>>0))
		fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #16\n", uint16(vaddr>>16))
		fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #32\n", uint16(vaddr>>32))
		fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #48\n", uint16(vaddr>>48))
		sb.WriteString("add x16, x19, x20\n")
		sb.WriteString("blr x16\n")
	}

	// Call the requested export (no args).
	fmt.Fprintf(&sb, "movz x20, #0x%X\n", uint16(exportVaddr>>0))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #16\n", uint16(exportVaddr>>16))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #32\n", uint16(exportVaddr>>32))
	fmt.Fprintf(&sb, "movk x20, #0x%X, lsl #48\n", uint16(exportVaddr>>48))
	sb.WriteString("add x16, x19, x20\n")
	sb.WriteString("blr x16\n")

	// exit_group(0)
	sb.WriteString("movz x8, #94\n")
	sb.WriteString("movz x0, #0\n")
	sb.WriteString("svc #0\n")

	engine, err := keystone.NewEngine(keystone.ARCH_ARM64, keystone.MODE_LITTLE_ENDIAN)
	if err != nil {
		return nil, err
	}
	defer func() { _ = engine.Close() }()
	return engine.Assemble(sb.String(), 0)
}
