/*
 *  Emulation of Linux signals
 *
 *  Copyright (c) 2003 Fabrice Bellard
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 */
#include "qemu/osdep.h"
#include "qemu.h"
#include "user-internals.h"
#include "signal-common.h"
#include "linux-user/trace.h"
#include "target_ptrace.h"

struct target_sigcontext {
    struct target_pt_regs regs;  /* needs to be first */
    uint32_t oldmask;
};

struct target_stack_t {
    abi_ulong ss_sp;
    int ss_flags;
    unsigned int ss_size;
};

struct target_ucontext {
    abi_ulong tuc_flags;
    abi_ulong tuc_link;
    target_stack_t tuc_stack;
    struct target_sigcontext tuc_mcontext;
    target_sigset_t tuc_sigmask;
};

/* Signal frames. */
struct target_rt_sigframe {
    target_siginfo_t info;
    struct target_ucontext uc;
    uint32_t tramp[2];
};

static void setup_sigcontext(struct target_sigcontext *sc, CPUMBState *env)
{
    for (int i = 0; i < 32; ++i) {
        __put_user(env->regs[i], &sc->regs.r[i]);
    }
    __put_user(env->pc, &sc->regs.pc);
}

static void restore_sigcontext(struct target_sigcontext *sc, CPUMBState *env)
{
    for (int i = 0; i < 32; ++i) {
        __get_user(env->regs[i], &sc->regs.r[i]);
    }
    __get_user(env->pc, &sc->regs.pc);
}

static abi_ulong get_sigframe(struct target_sigaction *ka,
                              CPUMBState *env, int frame_size)
{
    abi_ulong sp = env->regs[1];

    sp = target_sigsp(sp, ka);

    return ((sp - frame_size) & -8UL);
}

void setup_rt_frame(int sig, struct target_sigaction *ka,
                    target_siginfo_t *info,
                    target_sigset_t *set, CPUMBState *env)
{
    struct target_rt_sigframe *frame;
    abi_ulong frame_addr;

    frame_addr = get_sigframe(ka, env, sizeof *frame);
    trace_user_setup_rt_frame(env, frame_addr);

    if (!lock_user_struct(VERIFY_WRITE, frame, frame_addr, 0)) {
        force_sigsegv(sig);
        return;
    }

    frame->info = *info;

    __put_user(0, &frame->uc.tuc_flags);
    __put_user(0, &frame->uc.tuc_link);

    target_save_altstack(&frame->uc.tuc_stack, env);
    setup_sigcontext(&frame->uc.tuc_mcontext, env);

    for (int i = 0; i < TARGET_NSIG_WORDS; i++) {
        __put_user(set->sig[i], &frame->uc.tuc_sigmask.sig[i]);
    }

    /* Kernel does not use SA_RESTORER. */

    /*
     * Return from sighandler will jump to the tramp.
     * Negative 8 offset because return is rtsd r15, 8
     */
    env->regs[15] = default_rt_sigreturn - 8;

    /* Set up registers for signal handler */
    env->regs[1] = frame_addr;

    /* Signal handler args: */
    env->regs[5] = sig;
    env->regs[6] = frame_addr + offsetof(struct target_rt_sigframe, info);
    env->regs[7] = frame_addr + offsetof(struct target_rt_sigframe, uc);

    /* Offset to handle microblaze rtid r14, 0 */
    env->pc = (unsigned long)ka->_sa_handler;

    unlock_user_struct(frame, frame_addr, 1);
}


long do_sigreturn(CPUMBState *env)
{
    return -TARGET_ENOSYS;
}

long do_rt_sigreturn(CPUMBState *env)
{
    struct target_rt_sigframe *frame = NULL;
    abi_ulong frame_addr = env->regs[1];
    sigset_t set;

    trace_user_do_rt_sigreturn(env, frame_addr);

    if  (!lock_user_struct(VERIFY_READ, frame, frame_addr, 1)) {
        goto badframe;
    }

    target_to_host_sigset(&set, &frame->uc.tuc_sigmask);
    set_sigmask(&set);

    restore_sigcontext(&frame->uc.tuc_mcontext, env);

    target_restore_altstack(&frame->uc.tuc_stack, env);

    unlock_user_struct(frame, frame_addr, 0);
    return -QEMU_ESIGRETURN;

 badframe:
    unlock_user_struct(frame, frame_addr, 0);
    force_sig(TARGET_SIGSEGV);
    return -QEMU_ESIGRETURN;
}

void setup_sigtramp(abi_ulong sigtramp_page)
{
    uint32_t *tramp = lock_user(VERIFY_WRITE, sigtramp_page, 8, 0);
    assert(tramp != NULL);

    /*
     * addi r12, r0, __NR_rt_sigreturn
     * brki r14, 0x8
     */
    __put_user(0x31800000U | TARGET_NR_rt_sigreturn, tramp);
    __put_user(0xb9cc0008U, tramp + 1);

    default_rt_sigreturn = sigtramp_page;
    unlock_user(tramp, sigtramp_page, 8);
}
